import * as React from "react"
import { Slot } from "@radix-ui/react-slot"
import { cva, type VariantProps } from "class-variance-authority"

import { cn } from "@/lib/utils"

const buttonVariants = cva(
  "inline-flex items-center justify-center gap-2 whitespace-nowrap rounded-md text-sm font-medium transition-all disabled:pointer-events-none disabled:opacity-50 [&_svg]:pointer-events-none [&_svg:not([class*='size-'])]:size-4 shrink-0 [&_svg]:shrink-0 outline-none focus-visible:border-ring focus-visible:ring-ring/50 focus-visible:ring-[3px] aria-invalid:ring-destructive/20 dark:aria-invalid:ring-destructive/40 aria-invalid:border-destructive",
  {
    variants: {
      variant: {
        default:
          "bg-primary text-primary-foreground shadow-xs hover:bg-primary/90",
        destructive:
          "bg-destructive text-white shadow-xs hover:bg-destructive/90 focus-visible:ring-destructive/20 dark:focus-visible:ring-destructive/40 dark:bg-destructive/60",
        outline:
          "border bg-background shadow-xs hover:bg-accent hover:text-accent-foreground dark:bg-input/30 dark:border-input dark:hover:bg-input/50",
        secondary:
          "bg-secondary text-secondary-foreground shadow-xs hover:bg-secondary/80",
        ghost:
          "hover:bg-accent hover:text-accent-foreground dark:hover:bg-accent/50",
        link: "text-primary underline-offset-4 hover:underline",
        blue: "bg-gundam-blue text-primary-foreground shadow-xs hover:bg-gundam-blue/90",
        red: "bg-gundam-red text-primary-foreground shadow-xs hover:bg-gundam-red/90",
        black: "bg-gundam-black text-primary-foreground shadow-xs hover:bg-gundam-black/90",
        green: "bg-gundam-green text-primary-foreground shadow-xs hover:bg-gundam-green/90",
        semi: "bg-gundam-semi-black text-primary-foreground shadow-xs hover:bg-gundam-semi-black/90",
        gray: "bg-gundam-gray text-primary-foreground shadow-xs hover:bg-gundam-gray/90",
        menu: "bg-transparent text-white hover:bg-gundam-blue/90 focus-visible:ring-gundam-blue/20 shadow-xs justify-start text-md",
        active: "bg-gundam-blue text-white shadow-xs hover:bg-gundam-blue/90 justify-start text-md",
      },
      size: {
        default: "h-9 px-4 py-2 has-[>svg]:px-3",
        sm: "h-8 rounded-md gap-1.5 px-3 has-[>svg]:px-2.5",
        lg: "h-10 rounded-md px-6 has-[>svg]:px-4",
        icon: "size-9",
        menu: "h-10 w-full has-[>svg]:px-3 [&_svg]:size-6",
        bullet: "h-5 w-5 rounded-full p-1 has-[>svg]:px-1.5 cursor-pointer",
        rounded: "h-9 w-9 p-2 rounded-full has-[>svg]:px-2 cursor-pointer",
      },
    },
    defaultVariants: {
      variant: "default",
      size: "default",
    },
  }
)

function Button({
  className,
  variant,
  size,
  asChild = false,
  ...props
}: React.ComponentProps<"button"> &
  VariantProps<typeof buttonVariants> & {
    asChild?: boolean
  }) {
  const Comp = asChild ? Slot : "button"

  return (
    <Comp
      data-slot="button"
      className={cn(buttonVariants({ variant, size, className }))}
      {...props}
    />
  )
}

export { Button, buttonVariants }
