"use client"

import React from 'react'
import {
    flexRender,
    getCoreRowModel,
    getFilteredRowModel,
    getPaginationRowModel,
    getSortedRowModel,
    useReactTable,
} from "@tanstack/react-table"
import {
    Table,
    TableBody,
    TableCell,
    TableHead,
    TableHeader,
    TableRow,
} from "@/components/ui/table"
import { Button } from '@/components/ui/button';
import { Skeleton } from '@/components/ui/skeleton';
import { ChevronLeft, ChevronRight } from 'lucide-react'

interface Props {
    data: any[];
    columns: any;
    isLoading: boolean;
    next: () => void;
    prev: () => void;
    canPrevPage: boolean;
    canNextpage: boolean;
    page: number;
    allPage: number;
    total: number;
}

const DataTable = ({ data, columns, isLoading, next, prev, canPrevPage, canNextpage, page, allPage, total }: Props) => {
    const [columnVisibility, setColumnVisibility]= React.useState({})
    const [globalFilter, setGlobalFilter] = React.useState('')


    const column = React.useMemo(() => columns, [columns]);

    const table = useReactTable({
        data: data || [],
        columns: column,
        getCoreRowModel: getCoreRowModel(),
        getPaginationRowModel: getPaginationRowModel(),
        getSortedRowModel: getSortedRowModel(),
        getFilteredRowModel: getFilteredRowModel(),
        onColumnVisibilityChange: setColumnVisibility,
        onGlobalFilterChange: setGlobalFilter,
        state: {
            columnVisibility,
            globalFilter,
        },
    })

    return (
        <div className='space-y-6'>
            <Table className="border rounded-lg">
                <TableHeader className='bg-gray-300 h-12'>
                    {table.getHeaderGroups().map(headerGroup => (
                        <TableRow key={headerGroup.id}>
                            {headerGroup.headers.map(header => (
                                <TableHead key={header.id}>
                                {header.isPlaceholder
                                    ? null
                                    : flexRender(header.column.columnDef.header, header.getContext())
                                }
                                </TableHead>
                            ))}
                        </TableRow>
                    ))}
                </TableHeader>
                <TableBody>
                    {isLoading ? (
                        Array.from({ length: 10 }, (_, index) => (
                            <TableRow key={index} className="cursor-pointer" >
                                {Array.from({ length: columns.length }, (_, index2) => (
                                        <TableCell key={index2} >
                                            <Skeleton className="w-20 h-5 rounded-full" />
                                        </TableCell>
                                ))}
                            </TableRow>
                        ))
                    ) : (
                        table.getRowModel().rows.length > 0 ? (
                            table.getRowModel().rows.map(row => (
                                <TableRow key={row.id}>
                                    {row.getVisibleCells().map(cell => (
                                        <TableCell key={cell.id}>
                                        {flexRender(cell.column.columnDef.cell, cell.getContext())}
                                        </TableCell>
                                    ))}
                                </TableRow>
                            ))
                        ) : (
                            <TableRow>
                                <TableCell colSpan={columns.length} className="text-center">Data Empty</TableCell>
                            </TableRow>
                        )
                    )}
                </TableBody>
            </Table>
            <div className="flex items-center justify-end w-full">
                <div className="flex-1 text-sm text-muted-foreground">
                    Page {page} of{" "}
                    {allPage}
                    {" ( "}{total}{" ) "}
                </div>
                <div className='flex items-center gap-3'>
                    <Button
                        variant="outline"
                        className="text-xs"
                        size="sm"
                        onClick={() => prev()}
                        disabled={canPrevPage}
                    >
                        <ChevronLeft />
                        <span>Prev</span>
                    </Button>
                    <Button
                        variant="outline"
                        className="text-xs"
                        size="sm"
                        onClick={() => next()}
                        disabled={canNextpage}
                    >
                        <span>Next</span>
                        <ChevronRight />
                    </Button>
                </div>
            </div>
        </div>
    )
}

export default DataTable